--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Travel (and Disaster during travel)
	TODO: Separate disasters, like tips -- ?
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

gDestinationPort = nil

function Simulator:TravelTo(toPort)
	local route = gSim.port:GetRoute(gSim.port.name, toPort.name)
	local price = route.price or (route.totalWeeks * gSim.travelCost)
	if LQuest:Variable("airship") > 0 then price = 0 end
	if price > gSim.money then return false end

	gTravelActive = true
	gDestinationPort = toPort
	
	if route then
		-- Direct airship travel in half time
		if LQuest:Variable("airship") > 0 then
			local weeks = bsgFloor(route.totalWeeks / 2 + .5)
			local from = LPort.Locations[gSim.port.name]
			local to = LPort.Locations[toPort.name]
			route =
			{
				totalWeeks = weeks,
				{
					weeks = weeks,
					type = "airplane",
					{ from[1], from[2] },
					{ from[1], from[2] },
					{ to[1], to[2] },
					{ to[1], to[2] },
				}
			}
		end
		
		-- Decide whether a meeting or disaster will happen, and when
		
		-- Meeting odds: 20.0% chance per trip
		local meetingOdds = 200
		local meetingTime = route.totalWeeks + 1
	
		-- Disaster odds: 1/52 chance per week of travel, precedence over normal meetings
		local disasterOdds = 1000 * route.totalWeeks / 52	
		local disasterTime = route.totalWeeks + 1
		
		-- Force a meeting if there is a need to force a quest in _travel
		local b_travel = LBuilding:ByName("_travel")
		if (not gSim.quest) and b_travel and b_travel:ForceEncounter() then
--DebugOut("Forcing _travel encounter")
--DebugOut("week:"..gSim.weeks.." - questweek:"..gSim.questWeek)
			meetingOdds = 1000
			disasterOdds = 0
		end

		if gSim.rank > 0 and (gSim.weeks - gSim.disaster) > 40 and bsgRandom(1,1000) <= disasterOdds then
			-- Make the disaster mid-trip if possible (events happen at start of week)		
			if route.totalWeeks > 1 then disasterTime = bsgRandom(1,route.totalWeeks-1) + 1
			else disasterTime = 1
			end
		elseif gSim.rank > 0 and bsgRandom(1,1000) <= meetingOdds then
			-- Make the meeting mid-trip if possible (events happen at start of week)
			-- Dis-allow meetings on trips to/from secret ports
			if not (gSim.port.hidden or toPort.hidden) then
				if route.totalWeeks > 1 then meetingTime = bsgRandom(1,route.totalWeeks-1) + 1
				else meetingTime = 1
				end
			end
		end

		StartProductionAnim()
		
		local legsLeft = table.getn(route)
		local time = 0
		for _,leg in ipairs(route) do
			legsLeft = legsLeft - 1
			InitiateTravel(leg)
			for i=1,leg.weeks do
				time = time + 1
				if time == disasterTime then
					StopProductionAnim()
					self:Disaster()
					StartProductionAnim()
				elseif time == meetingTime then
					if b_travel then
						StopProductionAnim()
						b_travel:OnActivate()
						StartProductionAnim()
					end
				end
				
				-- On last week of final leg, queue port sting...
				if legsLeft == 0 and i == leg.weeks then
					SetMusicState("normal","music/"..toPort.name.."_sting.ogg")
				end
				
				TravelOneWeek(leg.type)
				Pause(1900)
				while StillTravelling() do Yield() end
				self:Tick()
				
				if self.stall and not self.stallwarn and gSim.rank > 0 then
					self.stallwarn = true
					SetLedgerContents("factories")
					DisplayDialog { "ui/okdialog.lua", body="factory_stalled", ledger=true }
				end
				
				StartProductionAnim()
			end
		end

		StopProductionAnim()
		
		-- Pay for the trip...
		if LQuest:Variable("airship") == 0 then
			local price = route.price or (route.totalWeeks * gSim.travelCost)
			if price > gSim.money then price = gSim.money end
			gSim:AdjustMoney(-price)
		end
	else
		bsgDevWarning("No route from "..gSim.port.name.." to "..toPort.name)
	end
	
	gTravelActive = nil
	gDestinationPort = nil
	self:ArriveAtPort(toPort)
	return true
end

function Simulator:ArriveAtPort(port)
	if type(port) == "string" then port = LPort:ByName(port) end
	
	self:SetPort(port)
	self:PrepareTravelPrices()
	gSim:QueueMessage(GetString("port_arrived", GetString(port.name)))
	
	-- Check for "all ports visited" medal in non-Free mode
	if not gSim.medals["ports"] and self.mode ~= "free" then
		local all = true
		for p in LPort:AllPorts() do
			if not p.hidden and not p.visited then
				all = false
				break
			end
		end
		if all then
			gNewMedal = "ports"
			self:AwardMedal("ports")
			UpdateBadge(self)
		end
	end

	-- On EnterPort, ledger will update if messages are visible, no need to force flush
	self:EnterPort()
end

function Simulator:SetPort(port)
	if type(port) == "string" then port = LPort:ByName(port) end

	-- As necessary, rebuild the colorized map
	if not port.visited then ResetColorMap() end

	-- Clearly port is available, now visited if not before
	port.available = true
	port.visited = true

	if self.port ~= port then
		self.port = port
		self.portName = port.name
		
		LItem:PreparePrices()
		LTip:ApplyAll()
	end
end

function Simulator:EnterPort()
	SwapToModal("ui/portview.lua")
end

-------------------------------------------------------------------------------
-- Routes

function DefineRoute(startName, endName, leg1, leg2)
	local r = LPort.Routes[startName]
	local totalWeeks = leg1.weeks
	if leg2 then totalWeeks = totalWeeks + leg2.weeks end
	if type(leg1) == "table" then r[endName] = { totalWeeks=totalWeeks, leg1, leg2 }
	elseif type(leg1) == "string" then r[endName] = { via=leg1 }
	else bsgDevWarning("Syntax problem in route from "..tostring(startName).." to "..tostring(endName))
	end
end

function DefaultRoute(startName, endName, weeks, type)
	local from = LPort.Locations[startName]
	local to = LPort.Locations[endName]
	
	weeks = weeks or 1
	type = type or "boat"
	
	local r = LPort.Routes[startName]
	r[endName] =
	{
		totalWeeks = weeks,
		{
			weeks = weeks,
			type = type,
			
			{ from[1], from[2] },
			{ from[1], from[2] },
			{ to[1], to[2] },
			{ to[1], to[2] },
		}
	}
	return r[endName]
end

function AppendRoute(route, startName, endName)
	local append = BuildCompleteRoute(startName, endName)
	if not append then
		bsgDevWarning("No route found from "..startName.." to "..endName)
	else
		route.compound = true
		route.totalWeeks = route.totalWeeks + append.totalWeeks
		for _,leg in ipairs(append) do
			table.insert(route, leg)
		end
	end
end

function ReverseRoute(reverseMe)
	local reversed =
	{
		totalWeeks = reverseMe.totalWeeks,
		compound = true,
	}
	
	for _,leg in ipairs(reverseMe) do
		local revLeg =
		{
			weeks = leg.weeks,
			type = leg.type
		}
		for _,point in ipairs(leg) do
			table.insert(revLeg,1,point)
		end
		
		table.insert(reversed,1,revLeg)
	end
	
	return reversed
end

function BuildCompleteRoute(startName, endName)
	local routeTable = LPort.Routes[startName]
	local targetRoute = routeTable[endName]

	if not targetRoute then
		-- Use a reverse routing
		targetRoute = BuildCompleteRoute(endName, startName)
		if not targetRoute then
			bsgDevWarning("No route found from "..endName.." to "..startName)
		else
			targetRoute = ReverseRoute(targetRoute)
		end
	elseif targetRoute.via then
		-- Expand a "via" routing
		local middleName = targetRoute.via
		targetRoute.via = nil
		targetRoute.totalWeeks = targetRoute.totalWeeks or 0
		AppendRoute(targetRoute, startName, middleName)
		AppendRoute(targetRoute, middleName, endName)
	end
	
	routeTable[endName] = targetRoute
	return targetRoute
end

-------------------------------------------------------------------------------
-- Disasters and helpers

local function RandomInventoryIngredient()
	local ingredients = {}
	for item in LItem:OwnedIngredients() do
		table.insert(ingredients, item)
	end
	if table.getn(ingredients) > 0 then
		local i = bsgRandom(1,table.getn(ingredients))
		return ingredients[i]
	else
		return nil
	end
end

local function RandomOwnedFactory()
	local factories = {}
	for f in LFactory:AllFactories() do
		if f.owned then table.insert(factories, f) end
	end
	if table.getn(factories) > 0 then
		local i = bsgRandom(1,table.getn(factories))
		return factories[i]
	else
		return nil
	end
end


Simulator._Disasters =
{
	-- Fire wipes out all inventory of a particular ingredient
	function()
		local i = RandomInventoryIngredient()
		if not i then return nil end
		i.inventory = 0
		gSim:InventoryChanged()
		
		gSim:Message(GetString("disaster_fire", GetString(i.name)))
		return GetString("disaster_fire_telegram", GetString(i.name))
	end,

	-- Favorable review. Product prices boosted for a particular product for N weeks.
	function()
		local tip = LTip:RandomTip("prod","up")
		gSim:Message(GetString("disaster_produp", GetString(tip.target), GetString(tip.port)))
		return GetString("disaster_produp_telegram", GetString(tip.port), GetString(tip.target))
	end,
	
	-- Factory strike requires player to reset production at one of their factories
	function()
		local f = RandomOwnedFactory()
		if not f then return nil end
		f:SetConfiguration(f.product.name, 0)
		LFactory:ProjectProduction()
		UpdateLedgerContents("factories")
		
		gSim:Message(GetString("disaster_strike", GetString(f.port.name)))
		return GetString("disaster_strike_telegram", GetString(f.port.name)), f.port.name, f.character[1]
	end,
	
	-- Inheritance. Receive some $ amount immediately.
	function()
		-- TODO: How much to inherit?
		local amount = gSim.rank * 100000
		gSim:AdjustMoney(amount)
		gSim:Message(GetString("disaster_inheritance", bsgDollars(amount)))
		return GetString("disaster_inheritance_telegram", bsgDollars(amount))
	end,
}

function Simulator:Disaster(force)
	local i = table.getn(self._Disasters)
	if force then i = bsgMod(force,i) + 1
	else i = bsgRandom(1,i)
	end
	
	local f = self._Disasters[i]
	if f and type(f) == "function" then
		local text,location,char = f()
		if text then
			gSim.disaster = gSim.weeks
		
			-- Default to sf_tutor, coming from Jakarta or SF depending on game stage (rank 3,4)
			if char then char = GetString(char.name)
			else char = GetString("sf_tutor")
			end
			if location then location = GetString(location)
			elseif gSim.rank >= 3 and gSim.rank <= 4 then location = GetString("jakarta")
			else location = GetString("sanfrancisco")
			end
			
			local dest = gDestinationPort
			if dest then dest = GetString(dest.name)
			else dest = GetString(gSim.port.name)
			end
			
			local date = bsgShortDate(gSim.weeks-1, 0)
			local time = { "1123AM", "0245PM", "0800AM", "0403PM" }
			date = date .. "  "..time[bsgRandom(1,table.getn(time))]
			
			local from = GetString("telegram_from", char).."<br>   "..location
			local to = GetString("telegram_to", gSim.player).."<br>   EN ROUTE "..dest
			
			local body = date.."<br>"..from.."<br><br>"..to.."<br><br>"..text
			DisplayDialog { "ui/telegram.lua", body=body }
			UpdateLedgerContents("messages")
		end
	end
end
